<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Cashbook extends Model
{
    protected $fillable = [
        'date',
        'type',
        'details',
        'amount',
        'balance',
        'warehouse_id',
        'user_id',
        'note',
        'reference_type',
        'reference_id',
        'is_opening_balance',
        'is_manual',
    ];

    protected $casts = [
        'date' => 'date',
        'amount' => 'decimal:2',
        'balance' => 'decimal:2',
    ];

    /**
     * Get the warehouse associated with the cashbook entry
     */
    public function warehouse()
    {
        return $this->belongsTo('App\Models\Warehouse');
    }

    /**
     * Get the user who created the cashbook entry
     */
    public function user()
    {
        return $this->belongsTo('App\Models\User');
    }

    /**
     * Scope to get debit entries (inward/received)
     */
    public function scopeDebits($query)
    {
        return $query->where('type', 'debit');
    }

    /**
     * Scope to get credit entries (outward/payments)
     */
    public function scopeCredits($query)
    {
        return $query->where('type', 'credit');
    }

    /**
     * Scope to filter by date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('date', [$startDate, $endDate]);
    }

    /**
     * Calculate running balance for all entries
     */
    public static function recalculateBalances($warehouseId = null)
    {
        $query = self::orderBy('date', 'asc')->orderBy('id', 'asc');
        
        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        $entries = $query->get();
        $balance = 0;

        foreach ($entries as $entry) {
            if ($entry->type === 'debit') {
                $balance += $entry->amount;
            } else {
                $balance -= $entry->amount;
            }
            
            $entry->balance = $balance;
            $entry->save();
        }
    }
}
