<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Payment;
use App\Models\Expense;
use App\Models\Income;
use App\Models\Returns;
use App\Models\ReturnPurchase;
use App\Models\Cashbook;
use App\Services\CashbookService;
use Illuminate\Support\Facades\DB;

class GenerateCashbookEntries extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cashbook:generate 
                            {--from-date= : Start date (Y-m-d format)}
                            {--to-date= : End date (Y-m-d format)}
                            {--warehouse= : Warehouse ID}
                            {--clear : Clear existing cashbook entries before generating}
                            {--opening-balance=0 : Set opening balance}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate cashbook entries from existing transactions (payments, expenses, income, returns)';

    protected $cashbookService;

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(CashbookService $cashbookService)
    {
        parent::__construct();
        $this->cashbookService = $cashbookService;
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Starting cashbook entry generation...');

        $fromDate = $this->option('from-date');
        $toDate = $this->option('to-date');
        $warehouseId = $this->option('warehouse');
        $clear = $this->option('clear');
        $openingBalance = $this->option('opening-balance');

        // Clear existing entries if requested
        if ($clear) {
            if ($this->confirm('Are you sure you want to clear all existing cashbook entries?')) {
                $query = Cashbook::query();
                if ($warehouseId) {
                    $query->where('warehouse_id', $warehouseId);
                }
                $deleted = $query->delete();
                $this->warn("Deleted {$deleted} existing cashbook entries.");
            } else {
                $this->info('Skipping clear operation.');
            }
        }

        // Set opening balance
        if ($openingBalance > 0) {
            $this->setOpeningBalance($openingBalance, $warehouseId);
        }

        $stats = [
            'payments' => 0,
            'expenses' => 0,
            'income' => 0,
            'sale_returns' => 0,
            'purchase_returns' => 0,
            'total' => 0,
        ];

        // Process Payments (cash only)
        $this->info('Processing payments...');
        $stats['payments'] = $this->processPayments($fromDate, $toDate, $warehouseId);

        // Process Expenses
        $this->info('Processing expenses...');
        $stats['expenses'] = $this->processExpenses($fromDate, $toDate, $warehouseId);

        // Process Income
        $this->info('Processing income...');
        $stats['income'] = $this->processIncome($fromDate, $toDate, $warehouseId);

        // Process Sale Returns
        $this->info('Processing sale returns...');
        $stats['sale_returns'] = $this->processSaleReturns($fromDate, $toDate, $warehouseId);

        // Process Purchase Returns
        $this->info('Processing purchase returns...');
        $stats['purchase_returns'] = $this->processPurchaseReturns($fromDate, $toDate, $warehouseId);

        $stats['total'] = array_sum($stats);

        // Recalculate all balances
        $this->info('Recalculating balances...');
        Cashbook::recalculateBalances($warehouseId);

        // Display summary
        $this->newLine();
        $this->info('=== Generation Complete ===');
        $this->table(
            ['Category', 'Entries Created'],
            [
                ['Payments (Cash)', $stats['payments']],
                ['Expenses', $stats['expenses']],
                ['Income', $stats['income']],
                ['Sale Returns', $stats['sale_returns']],
                ['Purchase Returns', $stats['purchase_returns']],
                ['TOTAL', $stats['total']],
            ]
        );

        $currentBalance = $this->cashbookService->getCurrentBalance($warehouseId);
        $this->info("Current Cash Balance: " . number_format($currentBalance, 2));

        return 0;
    }

    /**
     * Set opening balance
     */
    protected function setOpeningBalance($amount, $warehouseId = null)
    {
        $this->info("Setting opening balance: {$amount}");
        
        Cashbook::create([
            'date' => now()->subYears(10), // Far in the past
            'type' => 'debit',
            'details' => 'Opening Balance',
            'amount' => $amount,
            'balance' => $amount,
            'warehouse_id' => $warehouseId,
            'user_id' => 1, // System user
            'note' => 'System generated opening balance',
            'is_opening_balance' => true,
        ]);
    }

    /**
     * Process payments
     */
    protected function processPayments($fromDate, $toDate, $warehouseId)
    {
        $query = Payment::query();

        if ($fromDate) {
            $query->whereDate('created_at', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('created_at', '<=', $toDate);
        }

        // Only cash payments
        $query->whereIn('paying_method', ['cash', 'Cash']);

        $payments = $query->orderBy('created_at')->get();
        $count = 0;

        $this->output->progressStart($payments->count());

        foreach ($payments as $payment) {
            try {
                $this->cashbookService->createFromPayment($payment);
                $count++;
            } catch (\Exception $e) {
                $this->error("Failed to process payment #{$payment->id}: " . $e->getMessage());
            }
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
        return $count;
    }

    /**
     * Process expenses
     */
    protected function processExpenses($fromDate, $toDate, $warehouseId)
    {
        $query = Expense::query();

        if ($fromDate) {
            $query->whereDate('created_at', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('created_at', '<=', $toDate);
        }
        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        $expenses = $query->orderBy('created_at')->get();
        $count = 0;

        $this->output->progressStart($expenses->count());

        foreach ($expenses as $expense) {
            try {
                $this->cashbookService->createFromExpense($expense);
                $count++;
            } catch (\Exception $e) {
                $this->error("Failed to process expense #{$expense->id}: " . $e->getMessage());
            }
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
        return $count;
    }

    /**
     * Process income
     */
    protected function processIncome($fromDate, $toDate, $warehouseId)
    {
        $query = Income::query();

        if ($fromDate) {
            $query->whereDate('created_at', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('created_at', '<=', $toDate);
        }
        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        $incomes = $query->orderBy('created_at')->get();
        $count = 0;

        $this->output->progressStart($incomes->count());

        foreach ($incomes as $income) {
            try {
                $this->cashbookService->createFromIncome($income);
                $count++;
            } catch (\Exception $e) {
                $this->error("Failed to process income #{$income->id}: " . $e->getMessage());
            }
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
        return $count;
    }

    /**
     * Process sale returns
     */
    protected function processSaleReturns($fromDate, $toDate, $warehouseId)
    {
        $query = Returns::query();

        if ($fromDate) {
            $query->whereDate('created_at', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('created_at', '<=', $toDate);
        }
        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        $returns = $query->orderBy('created_at')->get();
        $count = 0;

        $this->output->progressStart($returns->count());

        foreach ($returns as $return) {
            try {
                $this->cashbookService->createFromSaleReturn($return);
                $count++;
            } catch (\Exception $e) {
                $this->error("Failed to process sale return #{$return->id}: " . $e->getMessage());
            }
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
        return $count;
    }

    /**
     * Process purchase returns
     */
    protected function processPurchaseReturns($fromDate, $toDate, $warehouseId)
    {
        $query = ReturnPurchase::query();

        if ($fromDate) {
            $query->whereDate('created_at', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('created_at', '<=', $toDate);
        }
        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        $returns = $query->orderBy('created_at')->get();
        $count = 0;

        $this->output->progressStart($returns->count());

        foreach ($returns as $return) {
            try {
                $this->cashbookService->createFromPurchaseReturn($return);
                $count++;
            } catch (\Exception $e) {
                $this->error("Failed to process purchase return #{$return->id}: " . $e->getMessage());
            }
            $this->output->progressAdvance();
        }

        $this->output->progressFinish();
        return $count;
    }
}
