<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Cashbook;
use App\Models\Warehouse;
use App\Services\CashbookService;
use App\Traits\StaffAccess;
use App\Traits\TenantInfo;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\Validator;
use Auth;
use DB;

class CashbookController extends Controller
{
    use StaffAccess;
    use TenantInfo;

    public function index(Request $request)
    {
        $role = Role::find(Auth::user()->role_id);
        if($role->hasPermissionTo('expenses-index')){
            $permissions = Role::findByName($role->name)->permissions;
            foreach ($permissions as $permission)
                $all_permission[] = $permission->name;
            if(empty($all_permission))
                $all_permission[] = 'dummy text';

            if($request->starting_date) {
                $starting_date = $request->starting_date;
                $ending_date = $request->ending_date;
            }
            else {
                $starting_date = date('Y-m-01');
                $ending_date = date("Y-m-d");
            }

            if($request->input('warehouse_id'))
                $warehouse_id = $request->input('warehouse_id');
            else
                $warehouse_id = 0;

            $lims_warehouse_list = Warehouse::select('name', 'id')->where('is_active', true)->get();
            
            // Calculate totals
            $query = Cashbook::whereDate('date', '>=' ,$starting_date)
                             ->whereDate('date', '<=' ,$ending_date);
            
            if($warehouse_id)
                $query->where('warehouse_id', $warehouse_id);
            
            $total_debit = $query->where('type', 'debit')->sum('amount');
            $total_credit = $query->where('type', 'credit')->sum('amount');
            
            // Get latest balance
            $latest_entry = Cashbook::orderBy('date', 'desc')->orderBy('id', 'desc')->first();
            $cash_on_hand = $latest_entry ? $latest_entry->balance : 0;

            return view('backend.cashbook.index', compact('lims_warehouse_list', 'all_permission', 'starting_date', 'ending_date', 'warehouse_id', 'total_debit', 'total_credit', 'cash_on_hand'));
        }
        else
            return redirect()->back()->with('not_permitted', __('db.Sorry! You are not allowed to access this module'));
    }

    public function cashbookData(Request $request)
    {
        $columns = array(
            1 => 'date',
            2 => 'details',
        );

        $warehouse_id = auth()->user()->warehouse_id ?? $request->input('warehouse_id');
        $q = Cashbook::whereDate('date', '>=' ,$request->input('starting_date'))
                     ->whereDate('date', '<=' ,$request->input('ending_date'));
        
        if($warehouse_id)
            $q = $q->where('warehouse_id', $warehouse_id);

        $totalData = $q->count();
        $totalFiltered = $totalData;

        if($request->input('length') != -1)
            $limit = $request->input('length');
        else
            $limit = $totalData;
        $start = $request->input('start');
        $order = 'cashbooks.'.$columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        
        if(empty($request->input('search.value'))) {
            $q = Cashbook::with('warehouse', 'user')
                ->whereDate('date', '>=' ,$request->input('starting_date'))
                ->whereDate('date', '<=' ,$request->input('ending_date'))
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir);
            
            if($warehouse_id)
                $q = $q->where('warehouse_id', $warehouse_id);
            $cashbooks = $q->get();
        }
        else
        {
            $search = $request->input('search.value');
            $q = Cashbook::with(['warehouse', 'user'])
                ->whereDate('cashbooks.date', '=', date('Y-m-d', strtotime(str_replace('/', '-', $search))))
                ->orWhere('cashbooks.details', 'LIKE', "%{$search}%")
                ->orWhere('cashbooks.amount', 'LIKE', "%{$search}%")
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir);
                
            if($warehouse_id)
                $q = $q->where('warehouse_id', $warehouse_id);
            $cashbooks = $q->get();

            $totalFiltered = $q->count();
        }

        $data = array();
        if(!empty($cashbooks))
        {
            foreach ($cashbooks as $key=>$cashbook)
            {
                $nestedData['id'] = $cashbook->id;
                $nestedData['key'] = $key;
                $nestedData['date'] = date('d-m-Y', strtotime($cashbook->date));
                $nestedData['type'] = $cashbook->type; // Keep lowercase for JS comparison
                $nestedData['details'] = $cashbook->details;
                $nestedData['amount'] = $cashbook->amount; // Keep as numeric for calculations
                $nestedData['amount_formatted'] = number_format($cashbook->amount, 2); // Add formatted version for display
                $nestedData['balance'] = $cashbook->balance; // Keep as numeric
                $nestedData['balance_formatted'] = number_format($cashbook->balance, 2); // Add formatted version
                $nestedData['warehouse'] = $cashbook->warehouse ? $cashbook->warehouse->name : 'N/A';
                $nestedData['options'] = '<div class="btn-group">
                            <button type="button" class="btn btn-default btn-sm dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">'.trans("db.action").'
                              <span class="caret"></span>
                              <span class="sr-only">Toggle Dropdown</span>
                            </button>
                            <ul class="dropdown-menu edit-options dropdown-menu-right dropdown-default" user="menu">
                                <li>
                                    <button type="button" data-id="'.$cashbook->id.'" class="open-EditCashbookDialog btn btn-link" data-toggle="modal" data-target="#editModal"><i class="dripicons-document-edit"></i> '.trans("db.edit").'</button>
                                </li>
                                <li class="divider"></li>
                                <li>
                                    <button type="button" class="delete-btn btn btn-link" data-id="'.$cashbook->id.'"><i class="dripicons-trash"></i> '.trans("db.delete").'</button>
                                </li>
                            </ul>
                        </div>';
                $data[] = $nestedData;
            }
        }
        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data
        );

        echo json_encode($json_data);
    }

    public function create()
    {
        //
    }

    public function store(Request $request)
    {
        $data = $request->all();
        
        $validator = Validator::make($data, [
            'date' => 'required|date',
            'type' => 'required|in:debit,credit',
            'details' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $data['user_id'] = Auth::id();
        $data['warehouse_id'] = $data['warehouse_id'] ?? null;
        
        $cashbook = Cashbook::create($data);
        
        // Recalculate balances
        Cashbook::recalculateBalances($data['warehouse_id']);

        return response()->json(['message' => 'Cashbook entry created successfully!']);
    }

    public function edit($id)
    {
        $cashbook = Cashbook::findOrFail($id);
        return $cashbook;
    }

    public function update(Request $request, $id)
    {
        $data = $request->all();
        
        $validator = Validator::make($data, [
            'date' => 'required|date',
            'type' => 'required|in:debit,credit',
            'details' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $cashbook = Cashbook::findOrFail($id);
        $warehouse_id = $cashbook->warehouse_id;
        
        $cashbook->update($data);
        
        // Recalculate balances
        Cashbook::recalculateBalances($warehouse_id);

        return response()->json(['message' => 'Cashbook entry updated successfully!']);
    }

    public function destroy($id)
    {
        $cashbook = Cashbook::findOrFail($id);
        $warehouse_id = $cashbook->warehouse_id;
        
        $cashbook->delete();
        
        // Recalculate balances after deletion
        Cashbook::recalculateBalances($warehouse_id);

        return response()->json(['message' => 'Cashbook entry deleted successfully!']);
    }

    public function deleteBySelection(Request $request)
    {
        $cashbook_id = $request['cashbookIdArray'];
        foreach ($cashbook_id as $id) {
            $cashbook = Cashbook::findOrFail($id);
            $cashbook->delete();
        }
        
        // Recalculate balances
        Cashbook::recalculateBalances();

        return 'Cashbook entries deleted successfully!';
    }

    /**
     * Set opening balance for cashbook
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function setOpeningBalance(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $warehouse_id = $request->input('warehouse_id');

        // Check if opening balance already exists
        $existing = Cashbook::where('is_opening_balance', true)
            ->where(function($q) use ($warehouse_id) {
                if ($warehouse_id) {
                    $q->where('warehouse_id', $warehouse_id);
                } else {
                    $q->whereNull('warehouse_id');
                }
            })
            ->first();

        if ($existing) {
            // Update existing opening balance
            $existing->update([
                'amount' => $request->amount,
                'date' => $request->date,
                'note' => $request->note ?? 'Opening balance',
            ]);
            
            Cashbook::recalculateBalances($warehouse_id);
            
            return response()->json(['message' => 'Opening balance updated successfully!']);
        }

        // Create new opening balance
        Cashbook::create([
            'date' => $request->date,
            'type' => 'debit',
            'details' => 'Opening Balance',
            'amount' => $request->amount,
            'balance' => $request->amount,
            'warehouse_id' => $warehouse_id,
            'user_id' => Auth::id(),
            'note' => $request->note ?? 'Opening balance',
            'is_opening_balance' => true,
        ]);

        Cashbook::recalculateBalances($warehouse_id);

        return response()->json(['message' => 'Opening balance set successfully!']);
    }

    /**
     * Generate cashbook entries from existing transactions
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function generateEntries(Request $request)
    {
        try {
            $cashbookService = app(CashbookService::class);
            
            // This will trigger the artisan command
            \Artisan::call('cashbook:generate', [
                '--from-date' => $request->from_date,
                '--to-date' => $request->to_date,
                '--warehouse' => $request->warehouse_id,
            ]);

            $output = \Artisan::output();

            return response()->json([
                'message' => 'Cashbook entries generated successfully!',
                'output' => $output
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error generating entries: ' . $e->getMessage()
            ], 500);
        }
    }
}
